/** @file   polygonobject.h
 * @brief   Declaration of PolygonObject - class.
 * @version $Revision: 1.1.1.1 $
 * @author  Tomi Lamminsaari
 */
 
#ifndef H_WWW_POLYGONOBJECT_H
#define H_WWW_POLYGONOBJECT_H

#include "gameobject.h"
#include "polygon.h"
#include <fstream>

namespace WeWantWar {


/** @class  PolygonObject
 * @brief   Extends the GameObject - class with some polygon-features.
 * @author  Tomi Lamminsaari
 *
 * This class provides simple interface for polygonized object. So far
 * the polygons can be rotated around the z-axis only and there is no
 * perspect projection. This just rotates the polygons and renders them
 * on given bitmap. The polygons can have textures though.
 *
 * The amount of rotation is taken from our <code> MobileObject </code> -
 * parent class.
 */
class PolygonObject : public GameObject
{
public:

  ///
  /// Constants, datatype and static members
  /// ======================================


  ///
  /// Constructors, destructor and operators
  /// ======================================

	/** Constructor.
   */
	PolygonObject();


	/** Destructor
   */
	virtual ~PolygonObject();

private:
	/** Copy constructor.
   * @param     rO                Reference to another PolygonObject
   */
  PolygonObject( const PolygonObject& rO );

	/** Assignment operator
   * @param     rO                Reference to another PolygonObject
   * @return    Reference to us.
   */
  PolygonObject& operator = ( const PolygonObject& rO );
  
public:


  ///
  /// Methods
  /// =======

  /** Redraws this object.
   * @param     pQueue            Pointer Redraw queue
   */
  virtual void redraw( RedrawQueue* pQueue );
  
  /** Adds new texture to this object. The texture bitmap is stored only
   * as a pointer so you should not destroy it if it's still in use. All
   * the textures this object has are stored in a table so you can
   * access them by their indices. Just remember the order you added
   * them to the object.
   * @param     pB                Pointer to texture bitmap.
   */
  void addTexture( BITMAP* pB );
  
  /** Changes the index'th texture.
   * @param     index             Index of the texture
   * @param     pB                Pointer to new texture bitmap
   */
  void setTexture( int index, BITMAP* pB );

  /** Adds new polygon to this object. The polygons added to this object
   * are owned by this PolygonObject and it destroys them in the destructor
   * so the polygons must be dynamically created. There is no upper limit
   * for the polygon count.
   * @param     pPoly             Pointer to polygon.
   */
  void addPolygon( Polygon* pPoly );

  /** Reads the data of this object from given input stream. This method
   * expects that the opening 'wewantwar_polygonobject' - tag has been
   * read away from the stream.
   * @param     rIn               The input stream where we read the data
   * @return    Nonzero if fails
   */
  int read( std::istream& rIn );

  ///
  /// Getter methods
  /// ==============
  
  /** Returns the index'th texture.
   * @param     index             Index of the texture being returned.
   * @return    Pointer to texture bitmap or null-pointer if there is
   *            no texture in such index.
   */
  BITMAP* getTexture( int index ) const;
  
  /** Returns the number of polygons this object has.
   * @return    Number of polygons
   */
  int polycount() const;

  /** Returns the pointer to requested polygon.
   * @param     index             Index of the Polygon being returned.
   * @return    Pointer to the index'th polygon or null-pointer if there is
   *            no Polygon in such index.
   */
  Polygon* getPolygon( int index ) const;

protected:

  ///
  /// Protected interface
  /// ===================
  
  /** Reads the polygon from given stream. The opening POLYGON - tag should
   * be read away from the stream
   * @param     rIn               The input stream
   * @return    Nonzero if fails.
   */
  int readPolygon( std::istream& rIn );
  
  /** Reads the data of a singe vertex
   * @param     rIn               The input stream
   * @param     pVtx              Pointer to vertex structure where we
   *                              read it.
   * @return    nonzero if fails
   */
  int readVertex( std::istream& rIn, Polygon::Vtx* pVtx );
  
  

  ///
  /// Members
  /// =======

  /** This vector holds all the polygons added to this object */
  std::vector< Polygon* > m_polytable;
  /** A table for the texture bitmaps. */
  std::vector< BITMAP* >  m_texturetable;

private:

  ///
  /// Private members
  /// ===============

};

};  // end of namespace

#endif

/**
 * Version history
 * ===============
 * $Log: polygonobject.h,v $
 * Revision 1.1.1.1  2006/01/21 23:02:42  lamminsa
 * no message
 *
 * Revision 1.0  2005-11-06 01:17:10+02  lamminsa
 * Initial revision
 *
 */
 
